﻿using Opc.UaFx;
using Opc.UaFx.Client;

namespace OpcUaClient
{
    // Autogenerated with OPC Watch
    [OpcDataTypeAttribute("ns=2;i=3002")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5001", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5003", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5002", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    public class KeyIntPair
    {
        public String? Key { get; set; }

        public Int32 Value { get; set; }
    }

    // Autogenerated with OPC Watch
    [OpcDataTypeAttribute("ns=2;i=3004")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5010", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5011", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    [OpcDataTypeEncodingAttribute("ns=2;i=5012", Type = Opc.UaFx.OpcEncodingType.Binary, NamespaceUri = "http://opcfoundation.org/UA/DI/")]
    public class KeyPosePair
    {
        public string? Key { get; set; }

        public int NoOfValue { get; set; }

        [OpcDataTypeMemberLengthAttribute(nameof(NoOfValue))]
        public double[]? Value { get; set; }
    }

    class Program
    {
        static void Main()
        {
            // Change IP address and login information
            string opcuaServerUrl = "opc.tcp://<robot-ip>:4840";
            string username = "username";
            string password = "password";

            OpcClient client = new OpcClient(opcuaServerUrl);
            client.Security.UserIdentity = new OpcClientIdentity(username, password);

            // Enforce an encrypted connection
            OpcSecurityPolicy securityPolicy = new OpcSecurityPolicy(OpcSecurityMode.SignAndEncrypt, OpcSecurityAlgorithm.Basic256Sha256);
            client.Security.EndpointPolicy = securityPolicy;

            client.UseDynamic = true;
            client.Connect();

            Console.WriteLine("Connection status: " + client.State + "\n");

            // Check if SPoC token is free
            OpcValue controlTokenActive = client.ReadNode("ns=2;i=7023");

            if (controlTokenActive.Status.IsGood && (bool)controlTokenActive.Value)
            {
                Console.WriteLine("Can not acquire SPoC token.");
                OpcValue controlTokenOwner = client.ReadNode("ns=2;i=7025");
                if (controlTokenOwner.Status.IsGood) Console.WriteLine("Robot is in use by: " + controlTokenOwner.Value);
                return;
            }

            // Get Status of brakes
            OpcValue brakeState = client.ReadNode("ns=2;i=6018");
            if (brakeState.Status.IsGood) Console.WriteLine("Checking Arm brake state: " + ((bool)brakeState.Value ? "Brakes are open" : "Brakes are closed"));

            // Get value for certain key in KeyIntMap
            Console.WriteLine("\n# Accessing the KeyIntMap");
            String key = "Franka Test Key";
            try
            {
                int value = (int)client.CallMethod("ns=2;i=5005", "ns=2;i=7002", key)[0];
                Console.WriteLine("  Getting value for key: '" + key + "', value: " + value);
            }
            catch (Opc.UaFx.OpcException ex)
            {
                if (ex.Code == OpcStatusCode.BadNotFound)
                {
                    Console.WriteLine("  [Warning] KeyIntMap does not contain an entry for '" + key + "'");

                }
                else
                {
                    Console.WriteLine("  [Error] Could not read " + key + " from KeyIntMap. Details: " + ex.Message);
                }
            }

            // Key Int Replace
            KeyIntPair keyIntPair = new KeyIntPair();
            keyIntPair.Key = key;
            keyIntPair.Value = 132;
            object[] keyIntReplaceResult = client.CallMethod("ns=2;i=5005", "ns=2;i=7001", keyIntPair);
            Console.WriteLine("  Writing KeyIntPair with key: '" + keyIntPair.Key + "' and value: " + keyIntPair.Value);

            // Read the previously set pair
            try
            {
                int value = (int)client.CallMethod("ns=2;i=5005", "ns=2;i=7002", keyIntPair.Key)[0];
                Console.WriteLine("  Getting value for key: '" + keyIntPair.Key + "', value: " + value);
            }
            catch (Opc.UaFx.OpcException ex)
            {
                Console.WriteLine("  [Error] Could not read " + keyIntPair.Key + " from KeyIntMap. Details: " + ex.Message);
            }

            // Read the current Cartesian pose
            Console.WriteLine("\n# Accessing runtime data");
            OpcValue result = client.ReadNode("ns=2;i=7020");
            if (result.Status.IsGood)
            {
                double[,] currentCartesianPose = (double[,])result.Value;
                double[] currentCartesianPose1D = new double[currentCartesianPose.GetLength(0) * currentCartesianPose.GetLength(1)];

                int index = 0;
                for (int i = 0; i < currentCartesianPose.GetLength(1); i++)
                {
                    for (int j = 0; j < currentCartesianPose.GetLength(1); j++)
                    {
                        currentCartesianPose1D[index] = currentCartesianPose[i, j];
                        index++;
                    }
                }

                Console.WriteLine("  Cartesian pose read at " + result.ServerTimestamp + ":\n  [" + String.Join(" ", currentCartesianPose1D) + "]");

                Console.WriteLine("\n# Accessing KeyPoseMap");

                // Write the current Cartesian Pose to the KeyPoseStore
                KeyPosePair keyPosePair = new KeyPosePair();

                keyPosePair.Key = "CartPose1";
                keyPosePair.Value = currentCartesianPose1D;
                keyPosePair.NoOfValue = currentCartesianPose1D.Length;

                object[] keyPoseReplaceResult = client.CallMethod("ns=2;i=5013", "ns=2;i=7011", keyPosePair);
                Console.WriteLine("  Wrote previously read Cartesian pose to key '" + keyPosePair.Key + "'");
            }

            client.Disconnect();
        }
    }
}
